/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2013 Tom Chothia, Yusuke Kawamoto and Chris Novakovic
 */

package bham.leakiest;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.PrintWriter;

import bham.leakiest.infotheory.MinEntropy;

/**
 * This class represents an RFC 4180-compliant CSV file containing information
 * about the leakage estimations generated by processing observations.
 * <br><br>
 * The columns of the CSV file are, in order:
 * <ol>
 *   <li>the number of observations processed so far;</li>
 *   <li>the uncorrected leakage estimation;</li>
 *   <li>the corrected leakage estimation;</li>
 *   <li>the lower bound on the confidence interval for the leakage estimate;</li>
 *   <li>the upper bound on the confidence interval for the leakage estimate;</li>
 *   <li>the upper bound on the value of corrected leakage that is consistent
 *       with zero leakage.</li>
 * </ol>
 * <br><br>
 * Not all of these statistics can be calculated for every leakage measure.
 * Statistics that are omitted are represented with the empty string in their
 * respective column.
 * 
 * @author Tom Chothia
 * @author Yusuke Kawamoto
 * @author Chris Novakovic
 * @version 1.4.3
 */
public class CSVFile {
	
	private PrintWriter printOut;

	/**
	 * Creates a new, empty CSV file at the path specified by <tt>out</tt>. If a
	 * file already exists at that path, it will be overwritten.
	 * @param out The path that the CSV file should be written to.
	 * @throws FileNotFoundException If <tt>out</tt> cannot be written to.
	 */
	public CSVFile(File out) throws FileNotFoundException {
		if (out == null) throw new FileNotFoundException("Invalid path to CSV file");
		
		printOut = new PrintWriter(new FileOutputStream(out));
		
		// Print CSV header
		printOut.println("Observations,Uncorrected leakage,Corrected leakage,Confidence interval lower bound,Confidence interval upper bound,Upper bound for zero leakage");
	}
	
	/**
	 * Adds a new line of information leakage statistics to the CSV file, given
	 * an Observations object. All statistics are calculated based on the data
	 * stored in this Observations object.
	 * @param observations The Observations object to use to calculate leakage
	 * statistics.
	 * @param leakageType The leakage measurement to calculate. Permitted values
	 * are:
	 * <ul>
	 *   <li><tt>1</tt> &mdash; mutual information;</li>
	 *   <li><tt>2</tt> &mdash; channel capacity;</li>
	 *   <li><tt>3</tt> &mdash; min-entropy;</li>
	 *   <li><tt>4</tt> &mdash; min-capacity.</li>
	 * </ul>
	 */
	public void addLeakageData(Observations observations, int leakageType) {
		//System.out.println("DEBUG: Calculating leakage after " + observations.getSampleCount() + " observations and writing to CSV file");
		
		double uncorrectedLeakage = Double.NaN,
			correctedLeakage = Double.NaN,
			confidenceIntervalLowerBound = Double.NaN,
			confidenceIntervalUpperBound = Double.NaN,
			upperBoundForZeroLeakage = Double.NaN;
		
		// An upper bound for zero leakage can only be calculated if a valid
		// channel is formed, i.e., one with more than one input and more than
		// one output
		boolean canCalculateUpperBound = observations.generateChannel().noOfInputs() > 1 && observations.generateChannel().noOfOutputs() > 1;			
		
		// Calculate the uncorrected/corrected values of leakage for the
		// required leakage measure
		switch (leakageType) {
			case TestInfoLeak.CLC_MUTUAL_INFO:
				uncorrectedLeakage = Estimate.getMutualInformation(observations);
				correctedLeakage = Estimate.getCorrectedMutualInformation(observations);
				confidenceIntervalLowerBound = canCalculateUpperBound ? Estimate.getCorrectedMutualInformationLowerBound(observations) : Double.NaN;
				confidenceIntervalUpperBound = canCalculateUpperBound ? Estimate.getCorrectedMutualInformationUpperBound(observations) : Double.NaN;
				upperBoundForZeroLeakage = canCalculateUpperBound ? Estimate.getUpperBoundForZeroLeakage(observations) : Double.NaN;
				break;
			case TestInfoLeak.CLC_CAPACITY:
				uncorrectedLeakage = Estimate.getCapacity(observations);
				correctedLeakage = Estimate.getCorrectedCapacity(observations);
				// TODO: do we need to calculate upper bound for capacity?
				break;
			case TestInfoLeak.CLC_MIN_ENTROPY:
				// Min-entropy leakage bounds can only be computed if we have
				// enough samples to estimate min-entropy leakage itself
				boolean sufficientSamples = observations.hasSufficientSamplesForMEL();
				
				//uncorrectedLeakage = Estimate.getMinEntropyLeak(observations);
				// This seems to be the new way to compute min-entropy leakage
				// estimates now
				uncorrectedLeakage = MinEntropy.minEntropyLeak(observations.getInputProbDist(), observations.generateChannel());
				
				confidenceIntervalLowerBound = sufficientSamples ? Estimate.getMinEntropyLeakLowerBound(observations) : Double.NaN;
				confidenceIntervalUpperBound = sufficientSamples ? Estimate.getMinEntropyLeakUpperBound(observations) : Double.NaN;
				break;
			case TestInfoLeak.CLC_MIN_CAPACITY:
				uncorrectedLeakage = Estimate.getMinCapacity(observations);
				// TODO: do we need to calculate corrected leakage and upper bound for min-capacity?
				break;
		}
		
		if (leakageType == TestInfoLeak.CLC_MUTUAL_INFO) {
			// We only calculate the confidence interval for mutual information
			// if the following conditions hold:
			// * if the uncorrected leakage value is above the upper bound for
			//   zero leakage;
			// * if the number of samples is at least 4 times greater than the
			//   product of the number of unique inputs and number of unique
			//   outputs
			boolean canCalculateConfidenceInterval =
				confidenceIntervalLowerBound != Double.NaN &&
				confidenceIntervalUpperBound != Double.NaN &&
				canCalculateUpperBound &&
				uncorrectedLeakage > upperBoundForZeroLeakage &&
				(4 * observations.getSampleCount() > observations.getUniqueInputCount() * observations.getUniqueOutputCount());
				
			if (!canCalculateConfidenceInterval) {
				confidenceIntervalLowerBound = Double.NaN;
				confidenceIntervalUpperBound = Double.NaN;
			}
		}
		
		addLeakageData(observations.getSampleCount(), uncorrectedLeakage, correctedLeakage, confidenceIntervalLowerBound, confidenceIntervalUpperBound, upperBoundForZeroLeakage);
	}
	
	/**
	 * Adds a new line of information leakage statistics to the CSV file, given
	 * all of the statistics that need to be written. <tt>Double.NaN</tt> values
	 * indicate that the statistic cannot be calculated, and are not written to
	 * the CSV file.
	 * @param observations The number of observations processed so far.
	 * @param uncorrectedLeakage The raw, uncorrected value for leakage.
	 * @param correctedLeakage The value of leakage after the correction has
	 * been applied.
	 * @param confidenceInterval The confidence interval for the corrected
	 * leakage estimation.
	 * @param upperBoundForZeroLeakage The upper bound for the value of
	 * corrected leakage that is consistent with zero leakage.
	 */
	public void addLeakageData(int observations, double uncorrectedLeakage, double correctedLeakage, double confidenceIntervalLowerBound, double confidenceIntervalUpperBound, double upperBoundForZeroLeakage) {
		printOut.println(
			(Double.isNaN(observations) ? "" : observations) + "," +
			(Double.isNaN(uncorrectedLeakage) ? "" : uncorrectedLeakage) + "," +
			(Double.isNaN(correctedLeakage) ? "" : correctedLeakage) + "," +
			(Double.isNaN(confidenceIntervalLowerBound) ? "" : confidenceIntervalLowerBound) + "," +
			(Double.isNaN(confidenceIntervalUpperBound) ? "" : confidenceIntervalUpperBound) + "," +
			(Double.isNaN(upperBoundForZeroLeakage) ? "" : upperBoundForZeroLeakage)
		);
	}
	
	
	/**
	 * Closes the CSV file. No more statistics may be written to this CSV file
	 * after this method has been called.
	 */
	public void close() {
		printOut.flush();
		printOut.close();
	}
	
}
