/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2013 Tom Chothia, Yusuke Kawamoto and Chris Novakovic
 */
package bham.leakiest;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import bham.leakiest.comparator.*;
import bham.leakiest.infotheory.*;

/**
 * This file contains the methods needed to turn a list of observations
 * in a conditional probability matrix and calculate the possible error.
 *
 * @author Tom Chothia
 * @author Yusuke Kawamoto
 * @author Chris Novakovic
 * @version 1.2
 */
public class Observations {
	//double Z = 1.96;  // The confidence in a single probability value 2.34 => 99.01 certainty
	//double certainty = 0.975;
	
	private double Z = 2.34;  // The confidence in a single probability value 2.34 => 99.01 certainty

	// Variables from TestInfoLeak
	private static int verbose = TestInfoLeak.verbose;

	/*
	 * Certainty
	 */
	double certainty = 0.999;
	
	private int noOfInputs = 0;
	private int noOfOutputs = 0;

	/*
	 * Mapping from unique input names to the number of times that input has
	 * been observed
	 */
	private LinkedHashMap<String, Integer> inputsSeen = new LinkedHashMap<String, Integer>();

	/*
	 * HashMap of unique output names (the Integer value is currently unused)
	 */
	private LinkedHashMap<String, Integer> outputsSeen = new LinkedHashMap<String, Integer>();
	
	/*
	 * The number of samples 
	 */
	private int noOfSamples = 0;

	/*
	 * The number of samples that had been supplied at the last time
	 * generateMatrix() was called; this allows repeated calls to the
	 * generateMatrix() method without having to do the expensive recomputation
	 * of the matrix
	 */
	private int noOfSamplesCached = 0;

	/*
	 * The number of samples given input
	 */
	int[] sampleCountGivenInput;

	/*
	 * The number of samples given output
	 */
	int[] sampleCountGivenOutput;

	/*
	 * The conditional probability matrix as a HashMap
	 */
	private LinkedHashMap<String, LinkedHashMap<String, Integer>> matrixMap = new LinkedHashMap<String, LinkedHashMap<String, Integer>>();

	/*
	 * Stores the observation matrix for the channel that is
	 * generated by this list of observations.
	 */
	private int[][] observationsMatrix;

	
	/*
	 * Stores the number of observing each input.
	 */
	private int[] inputObservationsArray;
	
	/*
	 * Stores the number of observing each output.
	 */
	private int[] outputObservationsArray;
	
	/*
	 * Stores the conditional probability matrix for the channel that is
	 * generated by this list of observations.
	 */
	private double[][] channelMatrix = new double[0][0];

	/**
	 * Constructs observations.
	 */
	public Observations() {}


	/**
	 * Adds input and output to observation.
	 * 
	 * @param input input value
	 * @param output output value
	 */
	public void addObservation(String input, String output) {
		// Increment the total number of samples
		noOfSamples++;
		
		Integer currentInputsSeen = inputsSeen.get(input);
		if(currentInputsSeen == null) { // If the input has never been seen before:
			// Increment the number of unique inputs seen
			noOfInputs++;
			
			// Initially set the number of times this input has been seen
			inputsSeen.put(input, 1);
			
			// If the input has never been seen before, the corresponding
			// output will never have been seen with this input either
			LinkedHashMap<String, Integer> newOutputMap = new LinkedHashMap<String, Integer>();
			newOutputMap.put(output, 1);
			matrixMap.put(input, newOutputMap);
		} else { // If the input has been seen before:
			// Increment the number of times this input has been seen
			inputsSeen.put(input, currentInputsSeen + 1);
			
			LinkedHashMap<String, Integer> matrixMapForInput = matrixMap.get(input);
			Integer currentOutputsForInput = matrixMapForInput.get(output);
			if(currentOutputsForInput == null) { // If the output has never been seen with this input before:
				// Initially set the number of times this output has been seen
				// with this input
				matrixMapForInput.put(output, 1);
			} else { // If the output has been seen with this input before:
				// Increment the number of times this output has been seen with
				// this input
				matrixMapForInput.put(output, currentOutputsForInput + 1);
			}
		}
		
		// Increment the number of unique outputs if this output has never been
		// observed before
		Integer currentOutputsSeen = outputsSeen.get(output);
		if(currentOutputsSeen == null) {
			outputsSeen.put(output, 1);
			noOfOutputs++;
		} else {
			outputsSeen.put(output, currentOutputsSeen + 1);
		}
	}

	
	/**
	 * Gives the channel matrix.
	 * @return channel matrix
	 */
	public double[][] getChannelMatrix() {
		// If the number of observations has not changed since the last time the
		// method was called, don't recompute the matrix - return the cached
		// version
		if(noOfSamplesCached == noOfSamples) return channelMatrix;
		
		// Otherwise, it needs to be recomputed
		generateMatrix();
		
		// channelMatrix now contains the channel matrix for the number of
		// observations in noOfSamples, so update the cache counter
		noOfSamplesCached = noOfSamples;
		
		return channelMatrix;
	}

	/**
	 * Gives the number of samples recorded by this Observations object so far.
	 * @return The number of samples recorded
	 */
	public int getSampleCount() {
		return noOfSamples;
	}
	
	/**
	 * Gives the numbers of samples that have a given input
	 * recorded by this Observations object so far.
	 * @return The numbers of samples recorded that have a given input
	 */
	public int[] getSampleCountGivenInput() {
		return sampleCountGivenInput;
	}
	
	/**
	 * Gives the numbers of samples that have a given output
	 * recorded by this Observations object so far.
	 * @return The numbers of samples recorded that have a given output
	 */
	public int[] getSampleCountGivenOutput() {
		return sampleCountGivenOutput;
	}
	
	/**
	 * Gives the number of unique pairs of inputs and outputs encountered so far.
	 * @return The number of unique pairs of inputs and outputs recorded by this {@link Observations}
	 * object.
	 */
	public int getDegreeOfFreedomMI() {
		if(observationsMatrix == null) {
			observationsMatrix = getObservationsMatrix();
		}
		// The number of non-zero elements in the joint matrix
		int counterJoint = 0;
		for(int row = 0; row < observationsMatrix.length; row++) {
			for(int col = 0; col < observationsMatrix[0].length; col++) {
				if(observationsMatrix[row][col] != 0) {
					counterJoint++;
				}
			}
		}
		
		// The number of non-zero elements in the input distribution
		int counterInput = 0;
		for(int row = 0; row < observationsMatrix.length; row++) {
			for(int col = 0; col < observationsMatrix[0].length; col++) {
				if(observationsMatrix[row][col] != 0) {
					counterInput++;
					break;
				}
			}
		}
		
		// The number of non-zero elements in the output distribution
		int counterOutput = 0;
		for(int col = 0; col < observationsMatrix[0].length; col++) {
		    for(int row = 0; row < observationsMatrix.length; row++) {
				if(observationsMatrix[row][col] != 0) {
					counterOutput++;
					break;
				}
			}
		}
		return counterJoint - counterInput - counterOutput + 1;
	}
	
	/**
	 * Gives the number of unique inputs encountered so far.
	 * @return The number of unique inputs recorded by this {@link Observations}
	 * object.
	 */
	public int getUniqueInputCount() {
		return noOfInputs;
	}
	
	/**
	 * Gives the number of unique outputs encountered so far.
	 * @return The number of unique outputs recorded by this
	 * {@link Observations} object.
	 */
	public int getUniqueOutputCount() {
		return noOfOutputs;
	}

	/**
	 * Gives the array of input names.
	 * @return the array of input names
	 */
	public String[] getInputNames()	{
		return inputsSeen.keySet().toArray(new String[0]);
	}

	/**
	 * Gives the array of output names.
	 * @return the array of output names
	 */
	public String[] getOutputNames() {
		return outputsSeen.keySet().toArray(new String[0]);
	}
	
	/*
	 * Returns the size/2 of the confidence interval of channelMatrix[x][y]
	 * with a certainty 99%.
	 * (Warning: assumes that <tt>input</tt> and <tt>output</tt> are both in
	 * <tt>matrixMap</tt>, so no <tt>null</tt>-checking is performed)
	 * 
	 * @param input input string 
	 * @param output output string
	 * @return the possible error with a certain confidence level
	 */
	private double calculateConfidence(String input, String output) {
		// We use the theory of "Confidence Intervals for Population Proportion"
		// i.e. true mean = p +/- Z.sqrt( (p(1-p))/n)
		// where p is the point estimate, n is the sample size
		//   and Z is the Z-value for the require confidence
		//   possible values include z = 2.34 for a confidence of.9901
		//                        or z = 3.49 for a confidence of .9998
		double n = inputsSeen.get(input).doubleValue();
		double p = matrixMap.get(input).get(output).doubleValue() / inputsSeen.get(input).doubleValue();
		double possError =  Z*  Math.sqrt( (p * (1-p))/n );
		//double possError =  Z*  Math.sqrt( (p * (1-p))/500000000 );
		
		//System.out.println("The 99% confidence interval of ("+x+","+y+") is ("+(p-possError)+","+(p+possError)+")");
		
		return possError;
	}
	
	/**
	 * Find the largest confidence interval.
	 *  
	 * @return the largest confidence interval
	 */
	public double largestInterval()	{
		double maxInterval = 0;
		for(String input : inputsSeen.keySet()) {
			for(String output : matrixMap.get(input).keySet()) {
				maxInterval = Math.max(maxInterval, calculateConfidence(input, output));
			}
		}
		return maxInterval;
	}
	
	/*
	public double maxErrorRatio() {
		double maxErrorRatio = 0;
		for(int x = 0; x < inputNames.size(); x++) {
			for(int y = 0; y < outputNames.size(); y++) {
				if(channelMatrix[x][y] != 0) {
					//System.out.println("Entry "+x+","+y+" has interval "+calculateConfidence(x,y)+" and value "+channelMatrix[x][y] );
					maxErrorRatio = Math.max(maxErrorRatio,
											 (calculateConfidence(x,y) / channelMatrix[x][y]) + 1);
				}
			}
		}
		return maxErrorRatio;
	}
	*/

	/*
	public double minErrorRatio() {
		double minErrorRatio = 1;
		for(int x = 0; x< inputNames.size(); x++) {
			for(int y = 0; y < outputNames.size(); y++) {
				if(channelMatrix[x][y] != 0) {
					//System.out.println("Looking at: "+x+","+y+": "+channelMatrix[x][y] +" with int. "+ calculateConfidence(x,y));
					//System.out.println(" ratio is:" + (1 - calculateConfidence(x,y) /channelMatrix[x][y] ));
					minErrorRatio = Math.min(minErrorRatio,
											 (1 - calculateConfidence(x,y) / channelMatrix[x][y]) );
				}
			}
		}
		return minErrorRatio;
	}
	*/
	
	/*
	public double totalCertainty() {
		return Math.pow(certainty, (inputNames.size() * (outputNames.size()-1)));
	}
	*/
	
	/**
	 * Tests whether the number of unique inputs recorded in this Observations
	 * object is sufficient for estimating mutual information. The minimum 
	 * required is 1.
	 * @return <tt>true</tt> if enough unique inputs have been collected to
	 * allow an accurate estimation of mutual information to be calculated;
	 * <tt>false</tt> if not.
	 */
	public boolean hasSufficientInputs() {
		return noOfInputs > 1;
	}
	
	/**
	 * Tests whether the number of unique outputs recorded in this Observations
	 * object is sufficient for estimating mutual information. The minimum 
	 * required is 1.
	 * @return <tt>true</tt> if enough unique outputs have been collected to
	 * allow an accurate estimation of mutual information to be calculated;
	 * <tt>false</tt> if not.
	 */
	public boolean hasSufficientOutputs() {
		return noOfOutputs > 1;
	}
	
	/**
	 * Tests whether the sample size recorded in this Observations object
	 * is sufficient for estimating mutual information.
	 * The minimum required sample size is <tt>i * o * 4</tt>, where 
	 * <tt>i</tt> is the number of unique inputs observed so far, and 
	 * <tt>o</tt> is the number of unique outputs observed so far.
	 * @return <tt>true</tt> if enough samples have been collected to allow an
	 * accurate estimation of mutual information to be calculated;
	 * <tt>false</tt> if not.
	 */
	public boolean hasSufficientSamplesForMI() {
		return noOfSamples > noOfInputs * noOfOutputs * 4;
	}


	/**
	 * Tests whether the sample size recorded in this Observations object
	 * is sufficient for estimating min-entropy leakage.
	 * We require that no more than 20% of the non-zero expected frequencies
	 * are below 5, while the expected frequencies are constructed from observed
	 * frequencies to find a confidence interval of min-entropy leakage.
	 * For efficiency of this test, we first check wether no more than 20%
	 * of the non-zero observed frequencies are below 5;
	 * if this is true, we next check wether no more than 20%
	 * of the non-zero expected frequencies are below 5.
	 * 
	 * @return <tt>true</tt> if enough samples have been collected to allow an
	 * accurate estimation of min-entropy leakage to be calculated;
	 * <tt>false</tt> if not.
	 */
	public boolean hasSufficientSamplesForMEL() {
		final int threshold  = 5;
		
		/* When the sample size is small, we efficiently checks
		   whether each observed frquency is 5 or more. */
		if(noOfSamples < threshold * noOfInputs * noOfOutputs) {
			return false;
		}
		
		/* When the sample size is not small, we check whether
		   no more than 20% of the non-zero observed frequencies are below 5. */
		int countSmallFrequencies = 0;
		int numCells = 0;
		for(String input : inputsSeen.keySet()) {
			for(String output : outputsSeen.keySet()) {
				Integer frequency = matrixMap.get(input).get(output);
				if(frequency != null && frequency != 0 && frequency < threshold) {
					countSmallFrequencies++;
				}
				numCells++;
			}
		}
		//System.out.println("Obs:409: countSmallFrequencies = " + countSmallFrequencies + " > " + ((double)numCells * 0.2) + " then sample size is not large enough.");
		if((double)countSmallFrequencies > (double)numCells * 0.2) {
			return false;
		}

		/* When the sample size is relatively large, we check whether
		   no more than 20% of the non-zero observed frequencies are below 5. */
		return MinEntropy.hasSufficientSamplesForMEL(this, numCells);
		/*
		countSmallFrequencies = 0;
		ArrayList<ArrayList<Integer>> list = InfoTheory.mergedData(this.getObservationsMatrix(), 0);
		final double chiSquarePosterior = Stats.chiSqu95Interval(InfoTheory.degreeOfFreedomPosterior(list));
		int[][] jodMin = InfoTheory.obsMinimizingCondVulnerability(list, chiSquarePosterior, noOfInputs, noOfOutputs);
		for(int numRow = 0; numRow < jodMin.length; numRow++) {
			for(int numCol = 0; numCol < jodMin[0].length; numCol++) {
				if(jodMin[numRow][numCol] < threshold) {
					countSmallFrequencies++;
				}
				if((double)countSmallFrequencies > (double)numCells * 0.2) {
					return false;
				}
			}
		}
		return true;
		*/
	}
	
	/**
	 * Generates the most probable channel matrix from the observations seen so
	 * far.
	 */
	private void generateMatrix() {		
		//System.out.println( "PS: size" + pairsSeen.size() + "PS element size "+pairsSeen.get(0).size());
		//System.out.println( "inputs  " + noOfInputs + "outputs: "+noOfOutputs);

		channelMatrix = new double[noOfInputs][noOfOutputs];

		int inputIndex = 0;
		for(String input : inputsSeen.keySet()) {
			int outputIndex = 0;
			for(String output : outputsSeen.keySet()) {
				Integer observationCount = matrixMap.get(input).get(output);
				channelMatrix[inputIndex][outputIndex++] = (observationCount != null ?
					matrixMap.get(input).get(output).doubleValue() / inputsSeen.get(input).doubleValue() :
					0);
			}
			inputIndex++;
		}

		// Counter the number of samples for each input
		sampleCountGivenInput = new int[noOfInputs];
		//System.out.println("noOfInputs = " + noOfInputs);
		String[] inputNames = this.getInputNames();
		for(int i = 0; i < this.noOfInputs; i++) {
			String input = inputNames[i];
			sampleCountGivenInput[i] = 0;
			//System.out.println("noOfOutputs = " + noOfOutputs);
			for(String output : outputsSeen.keySet()) {
				if(matrixMap.get(input).get(output) != null) {
					sampleCountGivenInput[i]+=	matrixMap.get(input).get(output);
				}
			}
		}
		
		// Counter the number of samples for each output
		sampleCountGivenOutput = new int[noOfOutputs];
		//System.out.println("noOfOutputs = " + noOfOutputs);
		String[] outputNames = this.getOutputNames();
		for(int i = 0; i < this.noOfOutputs; i++) {
			String output = outputNames[i];
			sampleCountGivenOutput[i] = 0;
			//System.out.println("noOfInputs = " + noOfInputs);
			for(String input : inputsSeen.keySet()) {
				if(matrixMap.get(input).get(output) != null) {
					sampleCountGivenOutput[i]+=	matrixMap.get(input).get(output);
				}
			}
		}
	}
	
	/**
	 * Like getChannelMatrix(), but the cells contain the number of times the
	 * respective input and output have been observed together.
	 * @return observations matrix
	 */
	public int[][] getObservationsMatrix() {
		this.observationsMatrix = new int[noOfInputs][noOfOutputs];
		
		int inputIndex = 0;
		//for(int i = 0; i < this.getInputNames().length; i++) {
		for(String input : inputsSeen.keySet()) {
			//String input = this.getInputNames()[i];
			int outputIndex = 0;
			for(String output : outputsSeen.keySet()) {
				Integer observationCount = matrixMap.get(input).get(output);
				observationsMatrix[inputIndex][outputIndex++] = (observationCount != null ? observationCount : 0);
			}
			inputIndex++;
		}
		return observationsMatrix;
	}

	/**
	 * Like getObservationsMatrix, but the cells are sorted according 
	 * the given arrays of input and output names.
	 * @param sortedInputNames
	 * @param sortedOutputNames
	 * @return sorted observations matrix
	 */
	public int[][] getSortedObservationsMatrix(String[] sortedInputNames, String[] sortedOutputNames) {
		if(this.observationsMatrix == null) {
			this.observationsMatrix = this.getObservationsMatrix();
		}
		int numRows = this.observationsMatrix.length;
		int numCols = this.observationsMatrix[0].length;
		if(this.observationsMatrix.length != sortedInputNames.length ||
		   this.observationsMatrix[0].length != sortedOutputNames.length ) {
    		System.out.print("Error: the length of the inputNames or outputNames does not match the size of the observation matrix:");
    		System.out.println("  observationsMatrix.length    = " + this.observationsMatrix.length);
    		System.out.println("  sortedInputNames.length      = " + sortedInputNames.length);
    		System.out.println("  observationsMatrix[0].length = " + this.observationsMatrix[0].length);
    		System.out.println("  sortedOutputNames.length     = " + sortedOutputNames.length);
    		System.exit(1);
		}
		
		int[][] sortedObservationsMatrix = new int[numRows][numCols];
		String[] inputNames = this.getInputNames();
		String[] outputNames = this.getOutputNames();
		for(int row = 0; row < numRows; row++) {
			int sortedRow = getIndex(sortedInputNames, inputNames[row]);
    		//System.out.println("  sortedRow = " + sortedRow + " <= inputNames[" + row + "] = " + inputNames[row]);
			for(int col = 0; col < numCols; col++) {
				int sortedCol = getIndex(sortedOutputNames, outputNames[col]);;
	    		//System.out.println("  sortedCol = " + sortedCol + " <= outputNames[" + col + "] = " + outputNames[col]);
				sortedObservationsMatrix[sortedRow][sortedCol] = this.observationsMatrix[row][col];
			}
		}
		return sortedObservationsMatrix;
	}

	/**
	 * Returns the hash map represetation of the observations matrix
	 * in which each cell contains the number of times the
	 * respective input and output have been observed together.
	 * 
	 * @return the hash map represetation of the observations matrix
	 */
	public LinkedHashMap<String, LinkedHashMap<String, Integer>> getObservationsMatrixMap() {
		return this.matrixMap;
	}
	
	/*
	 * Returns the index of the given name in the given array of labels.
	 * 
	 * @param labels array of labels
	 * @param name label
	 * @return index of the name in the labels
	 */
	private int getIndex(String[] labels, String name) {
		for(int i = 0; i < labels.length; i++) {
			if(name.equals(labels[i])) {
				return i;
			}
		}
		System.out.print("Error: the name " + name + " was not found in the array:");
		System.exit(1);
		return -1;
	}
	

	/**
	 * Returns the input probability distribution obtained from the observations.
	 * 
	 * @return the input probability distribution obtained from the observations
	 */
	public ProbDist getInputProbDist() {
		double[] inputProbs = new double[noOfInputs];
		String[] inputNames = new String[noOfInputs];
		
		int inputIndex = 0;
		for(String input : inputsSeen.keySet()) {
			int sum = 0;
			for(String output : outputsSeen.keySet()) {
				Integer observationCount = matrixMap.get(input).get(output);
				sum += (observationCount != null ? observationCount : 0);
			}
			inputProbs[inputIndex] = (double)sum / noOfSamples; 
			inputNames[inputIndex] = input;
			inputIndex++;
		}
		HashMap<State, Double> dist = new HashMap<State, Double>();
		for(int i = 0; i < noOfInputs; i++) {
			State st = new State();
			st.updateValue("input", inputNames[i]);
			//System.out.println("  sts[" + i + "] = " + stateNames[i]);
			double prob = inputProbs[i];
			//System.out.println("  pmf[" + i + "] = " + prob);
			dist.put(st, prob);
		}
		ProbDist pd = new ProbDist(dist, true);
		return pd;
	}
	
	
	/**
	 * Returns the array of numbers of input observations.
	 * 
	 * @return the array of numbers of input observations
	 */
	public int[] getInputObservationsArray() {
		this.inputObservationsArray = new int[noOfInputs];
		
		int inputIndex = 0;
		for(String input : inputsSeen.keySet()) {
			int sum = 0;
			for(String output : outputsSeen.keySet()) {
				Integer observationCount = matrixMap.get(input).get(output);
				sum += (observationCount != null ? observationCount : 0);
			}
			inputObservationsArray[inputIndex] = sum; 
			inputIndex++;
		}
		return inputObservationsArray;
	}

	/**
	 * Returns the array of numbers of output observations.
	 * @return the array of numbers of output observations
	 */
	public int[] getOutputObservationsArray() {
		this.outputObservationsArray = new int[noOfOutputs];
		
		int outputIndex = 0;
		for(String output : outputsSeen.keySet()) {
			int sum = 0;
			for(String input : inputsSeen.keySet()) {
				Integer observationCount = matrixMap.get(input).get(output);
				sum += (observationCount != null ? observationCount : 0);
			}
			outputObservationsArray[outputIndex] = sum; 
			outputIndex++;
		}
		return outputObservationsArray;
	}

	
	
	/**
	 * Returns the channel.
	 * 
	 * @return channel
	 */
	public Channel generateChannel() {
		return new Channel(Channel.BASIC, getInputNames(), getOutputNames(), getChannelMatrix());
	}

	
	/////////////////////////////////////////////////////////
	////// Transform observation counts to probabilities/////
	/**
	 * Returns the PMF array of the observations
	 * given an array of observed counts.
	 * 
	 * @param observedCounts array of observed counts 
	 * @return PMF array of the observations
	 */
	public static double[] observationsToPMF(int[] observedCounts) {
		int size = observedCounts.length;
		// Sum
		int sum = 0;
		for(int i = 0; i < size; i++) {
			sum += observedCounts[i];
		}
		// PMF
		double[] pmf = new double[size];
		for(int i = 0; i < size; i++) {
			pmf[i] = (double)observedCounts[i] / (double)sum;
		}
		return pmf;
	}
	
	
	/**
	 * Returns the joint probability matrix of the observations
	 * given a matrix of joint observed counts.
	 * 
	 * @param observationsMatrix observations matrix
	 * @return matrix of the joint probability matrix
	 */
	public static double[][] observationsToJPMF(int[][] observationsMatrix) {
		int noOfInputs = observationsMatrix.length;
		int noOfOutputs = observationsMatrix[0].length;
		// Sum
		int sum = 0;
		for(int i = 0; i < noOfInputs; i++) {
			for(int j = 0; j < noOfOutputs; j++) {
				sum += observationsMatrix[i][j];
			}
		}
		// PMF
		double[][] jointMatrix = new double[noOfInputs][noOfOutputs];
		for(int i = 0; i < noOfInputs; i++) {
			for(int j = 0; j < noOfOutputs; j++) {
				jointMatrix[i][j] = (double)(observationsMatrix[i][j]) / (double)sum;
			}
		}
		return jointMatrix;
	}
	
	
	////////////////////////////
	////// Print functions /////
	/**
	 * Print the observations matrix for the given observations file.
	 */
	public void printObservationsMatrix() {
		System.out.println("Observations matrix for the observations file:");
		int[][] observationsMatrix = this.getObservationsMatrix();
		String[] inputNames = this.getInputNames();
		String[] outputNames = this.getOutputNames();
		
		for (int i = 0; i < observationsMatrix.length; i++) {
			System.out.print(inputNames[i] + " -> ");
			for (int o = 0; o < observationsMatrix[i].length; o++) {
				System.out.print(outputNames[o] + "=" + observationsMatrix[i][o] + " ");
			}
			System.out.println();
		}
	}

	
	/*
	 * Variables for printJointFrequencyMatrix()
	 */
	//private ArrayList<StringWithSortIndex> outputNamesSorted = new ArrayList<StringWithSortIndex>();
	private ArrayList<Pair<String,Integer>> outputNamesSorted = new ArrayList<Pair<String,Integer>>();
	private final boolean sortOutputsDouble = true;
	private boolean sortOutputsDoubleDone = false;

	/**
	 * Print the joint probability matrix obtained from the given observations file.
	 */
	public void printJointFrequencyMatrix() {
		System.out.println("Joint probability matrix for the observations file:");
		this.getObservationsMatrix();
		String[] inputNames = this.getInputNames();
		String[] outputNames = this.getOutputNames();

		//System.out.println("  noOfSamples = " + noOfSamples);
		for (int i = 0; i < observationsMatrix.length; i++) {
			//System.out.print(inputNames[i] + " -> ");
			for (int o = 0; o < observationsMatrix[i].length; o++) {
				double prob = (double)observationsMatrix[i][o] / noOfSamples;
				//System.out.print(outputNames[o] + "=" + prob + " ");
				//System.out.print("(=" + observationsMatrix[i][o] + "/" + noOfSamples + ")  ");
			}
			//System.out.println();
		}
		//System.out.println("---------");

		// Sort the channel matrix by the output names
		if(sortOutputsDouble && !sortOutputsDoubleDone) {
			try {
				for(int i = 0; i < outputNames.length; i++) {
					//StringWithSortIndex swsi = new StringWithSortIndex();
					//swsi.setValues(outputNames[i], i);
					Pair<String,Integer> swsi = new Pair<String,Integer>(outputNames[i], i);
					outputNamesSorted.add(swsi);
				}
				ComparatorStringWithInt comparator = new ComparatorStringWithInt();
		        //ComparatorMulti comparator = new ComparatorMulti();
				Collections.sort(outputNamesSorted, comparator);
			} catch(Exception e) {
				System.out.println("Error: failed to sort outputs in sortByOutputValue(): " + e);
				System.exit(1);
			}
			sortOutputsDoubleDone = true;
		}
		
		// Find maximum output length
		int maxOutLength = 0;
		for(int i = 0; i < outputNames.length ; i++) {
			maxOutLength = Math.max(maxOutLength, outputNames[i].length());
		}
		//Rounding probs to 4 d.p. therefore all probs have length 6 
		maxOutLength = Math.max(maxOutLength,6);
		// Find the Max Input row label length
		int maxInLength = 0;
		for(int i = 0; i < inputNames.length; i++) {
			maxInLength = Math.max(maxInLength,inputNames[i].length());
		}
		maxInLength = maxInLength + 2;

		//Print the outputs names
		Channel.addspaces(maxInLength + 2);
		for(int i = 0; i < outputNames.length; i++) {
			System.out.print("| ");
			if(sortOutputsDouble) {
				//Channel.printToLength(outputNamesSorted.get(i).getOutputName(), maxOutLength);
				Channel.printToLength(outputNamesSorted.get(i).getElement1(), maxOutLength);
			}else {
				Channel.printToLength(outputNames[i], maxOutLength);
			}
		}
		System.out.println(""); 

		// Print each row of this channel matrix
		for(int i = 0; i < inputNames.length; i++)	{
			printRowJoint(i, maxInLength, maxOutLength, inputNames, outputNames);
		}
		System.out.println("");
		
		if(verbose >= 6) {
			printJointProbDist(inputNames, outputNames);
		}
		
	}

	/*
	 * Prints a row of the joint probability matrix obtained from this observation.
	 */
	private void printRowJoint(int RowNo, int maxInLength, int maxOutLength, String[] inputNames, String[] outputNames) {
		System.out.print(" ");
		Channel.printToLength(inputNames[RowNo], maxInLength+1); 
		for(int j = 0; j < outputNames.length; j++) { 
			System.out.print("| ");
			//int k = outputNamesSorted.get(j).getSortIndex();
			int k = outputNamesSorted.get(j).getElement2();
			//System.out.print("(j: " + j + "  k: " + k + ")");
			double prob = (double)observationsMatrix[RowNo][k] / this.noOfSamples;
			Channel.printToLength(Double.toString(Stats.round(prob,Math.max(4,maxOutLength-2))),maxOutLength);  
		}
		System.out.println("");
	}

	/*
	 * Prints the joint probability distribution of this observation (in the -prior format).
	 */
	private void printJointProbDist(String[] inputNames, String[] outputNames) {
		System.out.println("Joint probability distribution for the observations in the -prior format:");
		for(int RowNo = 0; RowNo < inputNames.length; RowNo++) { 
			for(int ColNo = 0; ColNo < outputNames.length; ColNo++) {
				//int k = outputNamesSorted.get(ColNo).getSortIndex();
				int k = outputNamesSorted.get(ColNo).getElement2();
				//System.out.print("(ColNo: " + ColNo + "  k: " + k + ")");
				double prob = (double)observationsMatrix[RowNo][k] / this.noOfSamples;
				System.out.print("(" + prob + ", (" + inputNames[RowNo]);	
				if(sortOutputsDouble) {
					//System.out.print(", " + outputNamesSorted.get(ColNo).getOutputName());
					System.out.print(", " + outputNamesSorted.get(ColNo).getElement1());
				} else {
					System.out.print(", " + outputNames[ColNo]);
				}
				System.out.println(")) ");
			}
		}
	}

	/*
	 * Prints the distribution.
	 */
	static void printDist(double[] array, String message) {
		int size = array.length;
		System.out.println(message + ":");
		System.out.print("{ ");
		for(int i = 0; i < size - 1; i++) { 
			System.out.println(array[i] + ",");
			System.out.print("  ");
		}
		System.out.println(array[size-1] + " }");
	}

	/*
	 * Prints the distribution.
	 */
	public static void printDist(int[] array, String message) {
		int size = array.length;
		System.out.println(message + ":");
		System.out.print("{  ");
		for(int i = 0; i < size - 1; i++) { 
			System.out.println(array[i] + ",");
			System.out.print("   ");
		}
		System.out.println(array[size-1] + "  }");
	}

	
	/*
	 * Prints the distribution.
	 */
	static void printDist(ArrayList array, String message) {
		int size = array.size();
		System.out.println(message + ":");
		System.out.print("{  ");
		for(int i = 0; i < size - 1; i++) { 
			System.out.println(array.get(i) + ",");
			System.out.print("   ");
		}
		System.out.println(array.get(size-1) + "  }");
	}

	
}
